<?php
/**
 * Main class
 *
 * @package woocommerce-print-invoice-delivery-notes
 */

/**
 * Exit if accessed directly
 */
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}


/**
 * Base class
 */
if ( ! class_exists( 'ar_invoice_wooCommerce' ) ) {
	/**
	 * WooCommerce Delivery Notes
	 *
	 * @author Tyche Softwares
	 * @package ar_invoices
	 */
	final class ar_invoice_wooCommerce {

		/**
		 * The single instance of the class
		 *
		 * @var object $ar_invoice_instance Instance object
		 */
		protected static $ar_invoice_instance = null;

		/**
		 * Sub class instances
		 *
		 * @var object $writepanel
		 */
		public $writepanel;

		/**
		 * Sub class instances
		 *
		 * @var object $print
		 */
		public $print;

		/**
		 * Sub class instances
		 *
		 * @var object $theme
		 */
		public $theme;

		/**
		 * Main Instance.
		 */
		public static function instance() {
			if ( is_null( self::$ar_invoice_instance ) ) {
				self::$ar_invoice_instance = new self();
			}
			return self::$ar_invoice_instance;
		}

		/**
		 * Constructor.
		 */
		public function __construct() {

			// WooCommerce activation required.
			include_once 'class-ar_invoices_instance-print.php';

			// Create the instances.
			$this->print = new ar_invoice_Print();

			add_action( 'wp_loaded', array( $this, 'load_hooks' ) );
			add_action( 'admin_init', array( $this, 'load_admin_hooks' ) );

			/**
			 * Return print page link
			 *
			 * @param array   $order_ids Order IDs.
			 * @param string  $template_type Template Type.
			 * @param string  $order_email Order email.
			 * @param boolean $permalink Permalinks.
			 */
			function ar_invoice_get_print_link( $order_ids, $template_type = 'order', $order_email = null, $permalink = false ) {
				global $ar_invoices_instance;
				return $ar_invoices_instance->print->get_print_page_url( $order_ids, $template_type, $order_email, $permalink );
			}

			/**
			 * Get orders as array. Every order is a normal WC_Order instance.
			 */
			function ar_invoice_get_orders() {
				global $ar_invoices_instance;
				return $ar_invoices_instance->print->orders;
			}

			/**
			 * Gets formatted item subtotal for display.
			 *
			 * @param object $order Order object.
			 * @param array  $item Item array.
			 * @param string $tax_display Display excluding tax or including.
			 */
			function ar_invoice_get_formatted_item_price( $order, $item, $tax_display = '' ) {
				if ( ! $tax_display ) {
					$tax_display = get_option( 'woocommerce_tax_display_cart' );
				}

				if ( ! isset( $item['line_subtotal'] ) || ! isset( $item['line_subtotal_tax'] ) ) {
					return '';
				}

				$wdn_order_currency = ( version_compare( get_option( 'woocommerce_version' ), '3.0.0', '>=' ) ) ? $order->get_currency() : $order->get_order_currency();

				if ( 'excl' === $tax_display ) {
					if ( version_compare( get_option( 'woocommerce_version' ), '3.0.0', '>=' ) ) {
						$ex_tax_label = wc_prices_include_tax() ? 1 : 0;
					} else {
						$ex_tax_label = $order->prices_include_tax ? 1 : 0;
					}

					$subtotal = wc_price(
						$order->get_item_subtotal( $item ),
						array(
							'ex_tax_label' => $ex_tax_label,
							'currency' => $wdn_order_currency,
						)
					);
				} else {
					$subtotal = wc_price( $order->get_item_subtotal( $item, true ), array( 'currency' => $wdn_order_currency ) );
				}

				return apply_filters( 'ar_invoice_formatted_item_price', $subtotal, $item, $order );
			}

		}

		public function load_admin_hooks() {
			// Hooks.
			add_action( 'woocommerce_admin_order_actions_end', array( $this, 'add_listing_actions' ) );
			add_action( 'add_meta_boxes', array( $this, 'add_box' ) );
			add_filter( 'bulk_actions-edit-shop_order', array( $this, 'register_my_bulk_actions' ) );
			add_filter( 'handle_bulk_actions-edit-shop_order', array( $this, 'my_bulk_action_handler' ), 10, 3 );
			add_filter( 'bulk_actions-woocommerce_page_wc-orders', array( $this, 'register_my_bulk_actions' ) );
			add_filter( 'handle_bulk_actions-woocommerce_page_wc-orders', array( $this, 'my_bulk_action_handler' ), 10, 3 );

			add_action( 'admin_notices', array( $this, 'confirm_bulk_actions' ) );
		}


		/**
		 * Load the hooks at the end when
		 * the theme and plugins are ready.
		 */
		public function load_hooks() {
			// hooks.
			add_filter( 'woocommerce_my_account_my_orders_actions', array( $this, 'create_print_button_account_page' ), 10, 2 );
			add_action( 'woocommerce_view_order', array( $this, 'create_print_button_order_page' ) );
			add_action( 'woocommerce_thankyou', array( $this, 'create_print_button_order_page' ) );

			add_action( 'woocommerce_email_after_order_table', array( $this, 'add_email_print_url' ), 100, 3 );
		}

		/**
		 * Is order edit page
		 */
		public function is_order_edit_page() {
			global $typenow, $pagenow;
			if ( 'shop_order' === $typenow && 'edit.php' === $pagenow ) {
				return true;
			} elseif ( isset( $_GET['page'] ) && 'wc-orders' === $_GET['page'] ) {
				return true;
			} else {
				return false;
			}
		}
		/**
		 * Add print actions to the orders listing
		 *
		 * @param object $order Order Object.
		 */
		public function add_listing_actions( $order ) {

			$wdn_order_id = ( version_compare( get_option( 'woocommerce_version' ), '3.0.0', '>=' ) ) ? $order->get_id() : $order->id;
			?>				
			<style>
				.print-preview-button{
					color: red;
					font-size: 18px;
					margin: 10px;
				}
			</style>
			<a href="<?php echo esc_url( ar_invoice_get_print_link( $wdn_order_id, 'invoice' ) ); ?>" class="tips print-preview-button <?php echo esc_attr( 'invoice' ); ?>" target="_blank" alt="<?php esc_attr_e( __( 'Print Invoice', THEME_NAME ) ); ?>" data-tip="<?php esc_attr_e( __( 'Print Invoice', THEME_NAME )); ?>">
				<i aria-hidden="true" class="isax isax-printer4"></i>
			</a>
			<?php
		}

		/**
		 * Add bulk actions to the dropdown.
		 *
		 * @param array $bulk_actions Array of the list in dropdown.
		 */
		public function register_my_bulk_actions( $bulk_actions ) {
			$bulk_actions['ar_invoice_print_invoice']       = apply_filters( 'ar_invoice_change_text_of_print_invoice_in_bulk_option', __( 'Print Invoice', THEME_NAME ) );
			return $bulk_actions;
		}

		/**
		 * Add bulk print actions to the orders listing
		 *
		 * @param string $redirect_to The redirect URL.
		 * @param string $doaction    The action being taken.
		 * @param array  $post_ids    Array of an IDs.
		 */
		public function my_bulk_action_handler( $redirect_to, $doaction, $post_ids ) {

			// stop if there are no post ids.
			if ( empty( $post_ids ) ) {
				return $redirect_to;
			}
			// stop if the action is not of bulk printing.
			if ( ! in_array( $_REQUEST['action'], array( 'ar_invoice_print_invoice' ) ) ) { // phpcs:ignore
				return $redirect_to;
			}

			if ( 'ar_invoice_print_invoice' === $doaction ) {
				$template_type = 'invoice';
				$report_action = 'printed_invoice';
			}
			if ( ! isset( $report_action ) ) {
				return $redirect_to;
			}

			// security check.
			if ( isset( $_GET['page'] ) && 'wc-orders' === $_GET['page'] ) {
				check_admin_referer( 'bulk-orders' );
			} else {
				check_admin_referer( 'bulk-posts' );
			}

			// get referrer.
			if ( ! wp_get_referer() ) {
				return $redirect_to;
			}

			// filter the referer args.
			$referer_args = array();
			parse_str( wp_parse_url( wp_get_referer(), PHP_URL_QUERY ), $referer_args );

			// set the basic args for the sendback.
			$args = array();
			if ( isset( $referer_args['post_status'] ) ) {
				$args = wp_parse_args( array( 'post_status' => $referer_args['post_status'] ), $args );
			}
			if ( isset( $referer_args['paged'] ) ) {
				$args = wp_parse_args( array( 'paged' => $referer_args['paged'] ), $args );
			}
			if ( isset( $referer_args['orderby'] ) ) {
				$args = wp_parse_args( array( 'orderby' => $referer_args['orderby'] ), $args );
			}
			if ( isset( $referer_args['order'] ) ) {
				$args = wp_parse_args( array( 'orderby' => $referer_args['order'] ), $args );
			}

			if ( isset( $referer_args['post_type'] ) ) {
				$args = wp_parse_args( array( 'post_type' => $referer_args['post_type'] ), $args );
			}

			if ( isset( $referer_args['page'] ) ) {
				$args = wp_parse_args( array( 'page' => $referer_args['page'] ), $args );
			}

			// do the action.
			$total = count( $post_ids );
			$url   = ar_invoice_get_print_link( $post_ids, $template_type );

			// generate more args and the sendback string.
			$args     = wp_parse_args(
				array(
					$report_action => true,
					'total'        => $total,
					'print_url'    => rawurlencode( $url ),
				),
				$args
			);
			$sendback = add_query_arg( $args, '' );
			wp_safe_redirect( $sendback );
			exit;
		}

		/**
		 * Show confirmation message that orders are printed
		 */
		public function confirm_bulk_actions() {
			if ( $this->is_order_edit_page() ) {
				if ( isset( $_REQUEST[ 'printed_invoice' ] ) ) {
					
					// use singular or plural form.
					$total   = isset( $_REQUEST['total'] ) ? absint( $_REQUEST['total'] ) : 0;
					$message = $total <= 1 ? $message = __( 'Invoice created.', THEME_NAME ) : __( 'Invoices created.', THEME_NAME );

					// Print URL - Fix Issue #214: Reflected XSS Vulnerability in Plugin.
					$print_url = isset( $_REQUEST['print_url'] ) ? $_REQUEST['print_url'] : ''; // phpcs:ignore
					$print_url = '' !== $print_url && strtolower( esc_url_raw( $print_url ) ) === strtolower( $print_url ) ? esc_url_raw( $print_url ) : '';

					if ( '' !== $print_url ) {
						?>
						<div id="ar_invoices-bulk-print-message" class="updated">
							<p><?php wp_kses_post( $message, THEME_NAME ); ?>
							<a href="<?php echo $print_url; // phpcs:ignore ?>" target="_blank" class="print-preview-button" id="ar_invoices-bulk-print-button"><?php esc_attr_e( 'Print now', THEME_NAME ); ?></a> <span class="print-preview-loading spinner"></span></p>
						</div>
						<?php
					}
				}
			}
		}

		/**
		 * Add the meta box on the single order page
		 */
		public function add_box() {
			if ( function_exists( 'wc_get_page_screen_id' ) ) {
				$screen = wc_get_page_screen_id( 'shop_order' );
			} else {
				$screen = 'shop_order';
			}
			add_meta_box( 'ar_invoices-box', __( 'Order Printing', THEME_NAME ), array( $this, 'create_box_content' ), $screen, 'side', 'low' );
		}

		/**
		 * Create the meta box content on the single order page
		 */
		public function create_box_content( $post ) {
			global $post_id, $ar_invoices_instance;

			$order_id = ( $post instanceof WP_Post ) ? $post->ID : $post->get_id();
			$order    = wc_get_order( $order_id );
			?>
			<div class="print-actions">
				<a href="<?php echo esc_url( ar_invoice_get_print_link( $order_id, 'invoice' ) ); ?>" class="button print-preview-button invoice" target="_blank" alt="<?php esc_attr_e( __( 'Print Invoice', THEME_NAME ) ); ?>"><?php esc_attr_e( 'Print Invoice', THEME_NAME ); ?></a>
			</div>
			<?php
		}

		/**
		 * Create a print button for the 'My Account' page.
		 *
		 * @param array  $actions My Account page actions.
		 * @param object $order Order object.
		 */
		public function create_print_button_account_page( $actions, $order ) {
			if ( '1' === option_value( 'ar_invoice_print_button_on_my_account_page' ) ) {
				// Add the print button.
				$wdn_order_id = ( version_compare( get_option( 'woocommerce_version' ), '3.0.0', '>=' ) ) ? $order->get_id() : $order->id;
				$actions['print'] = array(
					'url' => ar_invoice_get_print_link( $wdn_order_id, $this->get_template_type( $order ) ),
					'name' => apply_filters( 'ar_invoice_print_button_name_on_my_account_page', __( 'Print', THEME_NAME ), $order ),
				);
			}
			return $actions;
		}

		/**
		 * Create a print button for the 'View Order' page.
		 *
		 * @param int $order_id Order ID.
		 */
		public function create_print_button_order_page( $order_id ) {
			$order = new WC_Order( $order_id );
			$wdn_order_billing_id = ( version_compare( get_option( 'woocommerce_version' ), '3.0.0', '>=' ) ) ? $order->get_billing_email() : $order->billing_email;
			// Output the button only when the option is enabled.
			if ( '1' === option_value( 'ar_invoice_print_button_on_view_order_page' ) ) {
				// Default button for all pages and logged in users.
				$print_url = ar_invoice_get_print_link( $order_id, $this->get_template_type( $order ) );

				// Pass the email to the url for the tracking and thank you page. This allows to view the print page without logging in.
				if ( $this->is_woocommerce_tracking_page() ) {
					// changed.
					$wdn_order_email = isset( $_REQUEST['order_email'] ) ? sanitize_email( wp_unslash( $_REQUEST['order_email'] ) ) : '';
					$print_url = ar_invoice_get_print_link( $order_id, $this->get_template_type( $order ), $wdn_order_email );
				}

				// Thank you page.
				if ( is_order_received_page() && ! is_user_logged_in() ) {
					// Don't output the butten when there is no email.
					if ( ! $wdn_order_billing_id ) {
						return;
					}
					$print_url = ar_invoice_get_print_link( $order_id, $this->get_template_type( $order ), $wdn_order_billing_id );
				}

				?>
						<div class="d-flex justify-content-center">
							<a href="<?php echo esc_url( $print_url ); ?>" class="button print"><?php echo esc_attr( apply_filters( 'ar_invoice_print_button_name_order_page', __( 'Print', THEME_NAME ), $order ) ); ?></a>
						</div>
						<?php
			}
		}

		/**
		 * Add a print url to the emails that are sent to the customer
		 *
		 * @param object  $order Order Object.
		 * @param boolean $sent_to_admin Sent to admin true or false.
		 * @param boolean $plain_text Whether only to send plain text email or not.
		 */
		public function add_email_print_url( $order, $sent_to_admin = true, $plain_text = false ) {
			if ( '1' === option_value( 'ar_invoice_email_print_link' ) || '1' === option_value( 'ar_invoice_admin_email_print_link' ) ) {
				$wdn_order_billing_id = ( version_compare( get_option( 'woocommerce_version' ), '3.0.0', '>=' ) ) ? $order->get_billing_email() : $order->billing_email;

				$wdn_order_id = ( version_compare( get_option( 'woocommerce_version' ), '3.0.0', '>=' ) ) ? $order->get_id() : $order->id;

				$url = ar_invoice_get_print_link( $wdn_order_id, $this->get_template_type( $order ), $wdn_order_billing_id, true );

				if ( '1' === option_value( 'ar_invoice_email_print_link' ) ) {
					if ( ( $wdn_order_billing_id && ! $sent_to_admin ) ) {
						$this->print_link_in_email( $plain_text, $url );
					}
				}
				if ( '1' === option_value( 'ar_invoice_admin_email_print_link' ) ) {
					if ( $sent_to_admin ) {
						$this->print_link_in_email( $plain_text, $url );
					}
				}
			}
		}

		/**
		 * Html for Print Link in the emails.
		 *
		 * @param boolean $plain_text Whether only to send plain text email or not.
		 * @param string  $url Print Url in the email.
		 */
		public function print_link_in_email( $plain_text, $url ) {
			if ( $plain_text ) :
				echo esc_html_e( 'Print your order', THEME_NAME ) . "\n\n";

				echo esc_url( $url ) . "\n";

				echo "\n****************************************************\n\n";
			else :
				?>
						<p><strong><?php echo esc_attr_e( apply_filters( 'ar_invoice_print_text_in_email', 'Print:', THEME_NAME ) ); ?></strong> <a href="<?php echo esc_url_raw( $url ); ?>"><?php echo esc_attr_e( apply_filters( 'ar_invoice_print_view_in_browser_text_in_email', 'Open print view in browser', THEME_NAME ) ); ?></a></p>
				<?php endif;
		}

		/**
		 * Get the print button template type depnding on order status
		 *
		 * @param object $order Order object.
		 */
		public function get_template_type( $order ) {

			$wdn_order_status = ( version_compare( get_option( 'woocommerce_version' ), '3.0.0', '>=' ) ) ? $order->get_status() : $order->status;

			if ( 'completed' === $wdn_order_status ) {
				$type = apply_filters( 'ar_invoice_theme_print_button_template_type_complete_status', 'invoice' );
			} else {
				$type = apply_filters( 'ar_invoice_theme_print_button_template_type', 'order' );
			}

			$type = apply_filters( 'ar_invoice_theme_print_button_template_type_arbitrary', $type, $order );

			return $type;
		}

		/**
		 * Is WooCommerce 'Order Tracking' page
		 */
		public function is_woocommerce_tracking_page() {
			return ( is_page( wc_get_page_id( 'order_tracking' ) ) && isset( $_REQUEST['order_email'] ) ) ? true : false;
		}

	}
}
